<?php
namespace App\Controllers\Api\Ipn;

use App\Controllers\BaseController;
use App\Libraries\Uid;
use App\Models\AppsModel;
use App\Models\DepositMethodsModel;
use App\Models\PlansModel;
use App\Models\PlansExtraModel;
use App\Models\SubscribesModel;
use App\Models\SubscribesExtraModel;
use App\Models\TransactionsModel;
use App\Models\TransactionsExtraModel;
use CodeIgniter\HTTP\ResponseInterface;
use ReflectionException;
use Stripe\Exception\SignatureVerificationException;
use Stripe\Exception\UnexpectedValueException;
use Stripe\StripeClient;
use Stripe\Webhook;

require_once(APPPATH . 'ThirdParty/Stripe/init.php');

class Stripe extends BaseController
{
    public function index(): ResponseInterface
    {
        $methods = new DepositMethodsModel();
        $method = $methods->where("id", 1)->where("status", 1)->first();

        if (!$method) {
            return $this->respond(["message" => lang("Message.message_84")], 400);
        }

        if (empty($_SERVER['HTTP_STRIPE_SIGNATURE'])) {
            return $this->respond(["status" => "fail"], 400);
        }

        new StripeClient($method["api_value_1"]);

        $payload = @file_get_contents('php://input');
        $sig_header = $_SERVER['HTTP_STRIPE_SIGNATURE'];

        try {
            $event = Webhook::constructEvent($payload, $sig_header, $method["api_value_2"]);
        } catch (UnexpectedValueException $e) {
            return $this->respond(["message" => $e->getMessage()], 400);
        } catch (SignatureVerificationException $e) {
            return $this->respond(["message" => $e->getMessage()], 400);
        }

        $metadata = $event->data["object"]->metadata ?? null;

        if (!$metadata || !isset($metadata["type"])) {
            return $this->respond(["message" => "Metadata missing or type not specified"], 400);
        }

        switch ($event->type) {
            case 'checkout.session.completed':
                if ($metadata["type"] === 'plan') {
                    return $this->processPlan($event, $metadata);
                } elseif ($metadata["type"] === 'plan_extra') {
                    return $this->processPlanExtra($event, $metadata);
                }
                break;

            case 'invoice.paid':
                if ($metadata["type"] === 'plan') {
                    return $this->processInvoicePaid($event);
                } elseif ($metadata["type"] === 'plan_extra') {
                    return $this->processInvoicePaidExtra($event);
                }
                break;

            default:
                return $this->respond(["message" => "Unhandled event type"], 400);
        }

        return $this->respond(["status" => "ok"], 200);
    }

    private function processPlan($event, $metadata): ResponseInterface
    {
        $subscribes = new SubscribesModel();
        $uid = new Uid();

        $plan_id = $metadata["plan_id"];
        $user_id = $metadata["customer_id"];
        $app_id  = $metadata["app_id"];
        $amount_total = $event->data["object"]->amount_total / 100;

        $plans = new PlansModel();
        $plan = $plans->where("id", $plan_id)->select("count, build_count")->first();

        $currentDate = date('Y-m-d');
        $newDate = date('Y-m-d', strtotime('+' . $plan['count'] . ' month', strtotime($currentDate)));

        $subscribes->insert([
            "subscribe_external_id" => $event->data["object"]->subscription,
            "customer_external_id"  => $event->data["object"]->customer,
            "plan_id"               => $plan_id,
            "user_id"               => $user_id,
            "expires_at"            => strtotime($newDate),
            "app_id"                => $app_id,
            "price"                 => $amount_total,
            "uid"                   => $uid->create(),
            "is_active"             => 1,
            "remaining_count"       => $plan["build_count"],
            "method_id"             => 1
        ]);

        return $this->respond(["status" => "success"], 200);
    }

    private function processPlanExtra($event, $metadata): ResponseInterface
    {
        $subscribes = new SubscribesModel();
        $subscribesExtra = new SubscribesExtraModel();
        $uid = new Uid();

        $plan_extra_id = $metadata["plan_extra_id"];
        $user_id = $metadata["customer_id"];
        $app_id  = $metadata["app_id"];
        $amount_total = $event->data["object"]->amount_total / 100;

        // Sử dụng payment_intent thay vì customer khi customer không tồn tại
        $customer_extra_external_id = $event->data["object"]->customer ?? $event->data["object"]->payment_intent;
        
        if (!$customer_extra_external_id) {
            return $this->respond(["message" => "customer_extra_external_id is missing"], 400);
        }

        $plans_extra = new PlansExtraModel();
        $plan_extra = $plans_extra->where("id", $plan_extra_id)->select("build_count_extra")->first();

        // Cộng thẳng vào remaining_count trong bảng subscribes
        $subscribes->set('remaining_count', 'remaining_count + ' . $plan_extra["build_count_extra"], false)
                   ->where("user_id", $user_id)
                   ->where("app_id", $app_id)
                   ->update();

        // Insert new record into subscribes_extra
        $subscribesExtra->insert([
            "subscribe_extra_external_id" => $event->data["object"]->payment_intent,
            "customer_extra_external_id"  => $customer_extra_external_id,
            "plan_extra_id"               => $plan_extra_id,
            "user_id"                     => $user_id,
            "app_id"                      => $app_id,
            "price"                       => $amount_total,
            "uid"                         => $uid->create(),
            "method_id"                   => 1,
            "build_count_extra"           => $plan_extra["build_count_extra"]
        ]);

        return $this->respond(["status" => "success"], 200);
    }

    private function processInvoicePaid($event): ResponseInterface
    {
        $subscribes = new SubscribesModel();
        $uid = new Uid();

        $subscribe_external_id = $event->data["object"]->lines->data[0]->subscription;
        $expires_at = $event->data["object"]->lines->data[0]->period->end;
        $amount_paid  = $event->data["object"]->amount_paid / 100;
        $invoice_id  = $event->data["object"]->id;

        $subscribe = $subscribes->where("subscribe_external_id", $subscribe_external_id)->select("id")->first();

        if ($subscribe) {
            $subscribes->update($subscribe["id"], [
                "expires_at" => strtotime($expires_at),
            ]);
        }

        $transactions = new TransactionsModel();
        $transactions->insert([
            "uid"                   => $uid->create(),
            "amount"                => $amount_paid,
            "status"                => 1,
            "method_id"             => 1,
            "subscribe_external_id" => $subscribe_external_id,
            "external_uid"          => $invoice_id
        ]);

        return $this->respond(["status" => "success"], 200);
    }

    private function processInvoicePaidExtra($event): ResponseInterface
    {
        $transactionsExtra = new TransactionsExtraModel();
        $uid = new Uid();

        // Kiểm tra metadata để đảm bảo xử lý chính xác
        $metadata = $event->data["object"]->lines->data[0]->metadata ?? null;
        if (!$metadata || $metadata["type"] !== "plan_extra") {
            return $this->respond(["message" => "Metadata missing or type not specified"], 400);
        }

        $subscribe_extra_external_id = $event->data["object"]->payment_intent;
        $amount_paid = $event->data["object"]->amount_paid / 100;
        $invoice_id = $event->data["object"]->id;

        // Ghi lại giao dịch
        $transactionsExtra->insert([
            "uid" => $uid->create(),
            "amount" => $amount_paid,
            "status" => 1,
            "method_id" => 1,
            "subscribe_extra_external_id" => $subscribe_extra_external_id,
            "external_uid" => $invoice_id
        ]);

        return $this->respond(["status" => "success"], 200);
    }
}
